import json
import random

def load_data(filename):
    with open(filename, 'r') as file:
        data = json.load(file)
    messages = [entry['message'] for entry in data if entry['message']]
    return messages

def preprocess_message(message):
    return message.strip().lower()

def build_markov_chain(messages, chain_length=2):
    markov_chain = {}
    for message in messages:
        words = preprocess_message(message).split()
        for i in range(len(words) - chain_length):
            key = tuple(words[i:i + chain_length])
            next_word = words[i + chain_length]
            if key not in markov_chain:
                markov_chain[key] = []
            markov_chain[key].append(next_word)
    return markov_chain

def generate_response(markov_chain, prompt, length=10):
    prompt_words = preprocess_message(prompt).split()
    start_key = tuple(prompt_words[-2:]) if tuple(prompt_words[-2:]) in markov_chain else random.choice(list(markov_chain.keys()))
    message = list(start_key)
    for _ in range(length - len(start_key)):
        if start_key in markov_chain:
            next_word = random.choice(markov_chain[start_key])
            message.append(next_word)
            start_key = tuple(message[-2:])
        else:
            break
    return ' '.join(message).capitalize() + '.'

def main():
    messages = load_data('data.json')
    markov_chain = build_markov_chain(messages)
    
    print("Start a conversation (type 'exit' to quit):")
    while True:
        prompt = input("You: ")
        if prompt.lower() == 'exit':
            break
        response = generate_response(markov_chain, prompt)
        print("AI:", response)

if __name__ == "__main__":
    main()
